function [exp_notional, exp_notional_loss] = expected_index_notional(dates, T, portfolio, index_members, maturities, ajd_common, y0)
% --------------------------------------------------------------------------------------------------
% Calculate the expected index notional and loss of notional for various time horizons. This
% function is fully vectorized: each row of the output corresponds to a different date.
% --------------------------------------------------------------------------------------------------
% dates             ... dates for which to compute future expected index notional (loss)
% T                 ... horizons for which to compute future expected index notional (loss)
% portfolio         ... portfolio of interest
% index_members     ... indicator matrix, i-th row representing members of index for i-th date
% maturities        ... which maturities to update (indicator vector), default: update all
% ajd_common        ... dynamics of common factor
% y0                ... initial value(s) of common factor
% --------------------------------------------------------------------------------------------------
% sample call: expected_index_notional(cdx_test.dates{1}, cdx_test.T{1}, cdx_test.portfolio, cdx_test.index_members)
%              expected_index_notional(cdx_mor.dates{1}(1), cdx_mor.T{1}, cdx_mor.portfolio, cdx_mor.index_members)
% --------------------------------------------------------------------------------------------------

% Compute expected quarterly portfolio notional and notional loss;
num_firms = length(portfolio);
num_dates = length(dates);
T_max = ceil(max(T)*4)/4;
surv_probs = ones(num_firms, num_dates, T_max*4);
num_missing = zeros(num_dates, 1);
for i=1:num_firms
    % Calculate default probabilities for i-th CDS for each horizon
    default_probs = default_probs_cds2(portfolio(i), maturities, dates(1), dates(end), ajd_common, y0);
    
    % Impute into 3-dim matrix by matching dates, keep track of missing data
    [found, pos_index, pos_cds] = intersect_sorted(dates, default_probs.dates);
    not_found = find(~found);
    num_missing(not_found) = num_missing(not_found) + 1;
    if (sum(found)>0)
        dim = size(default_probs.surv_probs, 2);
        surv_probs(i,pos_index,1:min(T_max*4, dim)) = default_probs.surv_probs(pos_cds,1:(min(T_max*4, dim)));
    end
end

% Convert survival probabilities into expected index notional (loss)
exp_notional = zeros(num_dates, T_max*4);
for t=1:(T_max*4)
    for j=1:num_dates
        used_members = logical(index_members(j,:));
        exp_notional(j,t) = sum(squeeze(surv_probs(used_members,j,t)))'/sum(used_members);
    end
end
exp_notional_loss = [ones(num_dates, 1) exp_notional(:,1:end-1)] - exp_notional;

% Scale up losses by fraction of missing credit default swaps
exp_notional_loss = exp_notional_loss .* repmat(1./(1-num_missing/125), 1, T_max*4);
exp_notional = 1 - (1-exp_notional) .* repmat(1./(1-num_missing/125), 1, T_max*4);
